/*
 * Adonis project TCL software utilities:
 *  Binary data handling in tcl
 *  written by L. Demailly
 *  (c) 1994-1996 Observatoire de Paris - Meudon - France
*/
/*
# For extraction by AutoDoc's c2tcldoc / document
# $Id: bindata.c,v 2.15 1997/01/29 23:39:44 dl Exp $ 
*/
/* Description : associate C usable binary objects to tcl usable arrays
 *
 * NB: this .c file is large because it contains the documentation emmbeded
 *               
 * $Log: bindata.c,v $
 * Revision 2.15  1997/01/29 23:39:44  dl
 * added -type option to bin_info (to get only object's type)
 * fixes for commands usage error messages when flags are present (argv++..)
 *
 * Revision 2.14  1997/01/27 22:14:05  dl
 * possible bug fix in case of empty type names (bin_new)
 * fixes in documentation
 * FEATURE EXTENSION: bin_def now has a nelems arg like the C api
 * can be used to create array fields. Added special support for
 * char array fields. should replace cleanly 'str' built in type.
 *
 * Revision 2.13  1997/01/27 00:22:45  dl
 * updated author section (email/url)
 *
 * Revision 2.12  1997/01/26 23:54:20  dl
 * added memset like option to bin_copy (when source is an int instead of
 * object). bin_copy now returns the number of bytes copied.
 *
 * Revision 2.11  1997/01/26 14:45:21  dl
 * added missing offset parameter in autodoc for bin_def (thx tip)
 *
 * Revision 2.10  1996/06/25 13:17:13  dl
 * added doc infos on Bin_GenerateName limitations
 *
 * Revision 2.9  1996/06/25  08:12:47  dl
 * bug fix in Bin_GenerateName (could fail after 99999 calls with large basename)
 * added a lot of needed CONST, but they generate compile warnings because
 * of Tcl missing consts... added a #ifdef DISCARD_CONST to avoid this in
 * bindata.h
 *
 * Revision 2.8  1996/05/30  10:56:26  dl
 * new Bin_SizeofType
 *
 * Revision 2.7  1996/05/10  20:44:14  dl
 * avoid null pointer indirection in fprintf when name2 is NULL (solaris)
 *
 * Revision 2.6  1996/05/10  20:05:50  dl
 * renamed Bin_NameObject into Bin_TclAttach which also perform the
 * unset callback setting to delete child objects when father object is unset
 * cleaner simpler Bin_NewCmd and api (thx again ztf for suggesting this move)
 *
 * Revision 2.5  1996/05/10  16:54:36  dl
 * documenation improvements
 *
 * Revision 2.4  1996/05/09  16:48:06  dl
 * Uh uh... Bin_NameObject bug fix (_obj_ was set to obj name instead of type)
 * one more cheerfull thank you to ztf for pointing it out !
 *
 * Revision 2.3  1996/05/09  15:47:01  dl
 * pointer casts to avoid compiler warnings
 *
 * Revision 2.2  1996/05/08  20:40:35  dl
 * Inline documentation completed. Cleanup in presentation. Debug output on s
 * stderr.
 *
 * Revision 2.1  1996/05/07  22:18:25  dl
 * Ident cleanup. Started emmbeded auto documentation.
 * Added Bin_NameObject (ztf suggestion).
 *
 * Revision 1.23  1996/05/06  21:09:45  dl
 * some minor indent fixes. clientdata comment fixup : either NULL (not used)
 * or in fact a Tcl_HashTable pointing to the registered types
 * Bin_Def calls Bin_GetTHT if passed a NULL table.
 *
 * Revision 1.22  1996/05/06  19:02:22  dl
 * new Bin_GetTHT . can now move met_new creation outside Bin_Init
 *
 * Revision 1.21  1996/05/03  14:33:51  dl
 * added pointer type cast to avoid compiler warnings
 *
 * Revision 1.20  1996/05/02  23:56:02  dl
 * using standard autoconf defines for ltoa and includes
 *
 * Revision 1.19  1996/05/02  20:31:25  dl
 * We register as a package, for tcl7.5 (suggested by ztf)
 *
 * Revision 1.18  1996/05/02  11:40:36  dl
 * added netint & netshort built in types and traces (net order 32/16 bits)
 *
 * Revision 1.17  1996/04/24  11:52:22  dl
 * bin_info now generates slightly better keyed list (type can be "")
 *
 * Revision 1.16  1996/04/24  09:23:24  dl
 * -nobuf, for tcl7.5 IOs for bin_read is not supported.
 *
 * Revision 1.15  1996/04/23  14:25:37  dl
 * tcl7.5 compatibility for IO (using new Tcl_Channels) - conditional compile
 *
 * Revision 1.14  1996/04/19  11:53:39  dl
 * documentation/comments fix ups
 *
 * Revision 1.13  1995/12/11  12:39:56  dl
 * added -swapb and -rev4 option (for machine dep cvrt) to bin_copy
 *
 * Revision 1.12  1995/12/04  09:35:29  dl
 * bug fix in IO error messages in -nobuf case (thx Martin Cornelius)
 *
 * Revision 1.11  1995/11/16  01:50:30  dl
 * changed what should have been the last possible out of bounds
 * access, for the "str" type, read was reading 'till a 0 is found,
 * it is now copied to a null terminated buffer first.
 *
 * Revision 1.10  1995/10/31  10:36:11  dl
 * Memory leak in Complex trace fixed (thx Purify !)
 *
 * Revision 1.9  1995/09/29  15:58:01  dl
 * bin_copy now uses memmove so areas can overlap (sub pointers objects)
 * bin_read and and bin_write have a -nobuf optional flag to use directly
 * read/write syscall, skipping completly stdio buffering
 * added missing #includes
 * complex numbers type support moved to conditional ADONIS compile
 *
 * Revision 1.8  1995/08/11  14:39:52  dl
 * minor cast and ltoa change
 *
 * Revision 1.7  1995/08/10  12:32:00  dl
 * bin_read/write was returning number of object instead of bytes (0/1)
 * added bytes read/written in error message for bin_read/write
 *
 * Revision 1.6  1995/08/10  11:57:07  dl
 * Added bin_resize (thx to Dimitry Kloper dimka@tochna.technion.ac.il)
 * bin_read and bin_write now returns # of bytes actually read/written
 * casts added to avoid compiler complaints
 * using long format in sprintf for long...
 * added an ltoa replacement (use -DNEED_LTOA if you don't have it) (Dimitry)
 *
 * Revision 1.5  1994/12/31  19:58:00  dl
 * added complex type, info command, father in bin_new (ptr)
 * changed the type* management
 * added -absolute and -noerr flags to bin_move
 * removed the _obj offset_... tcl fields of objects... (all this is
 * avaliable through bin_info now)
 * added a C api bin_def
 * set obj(i) works for every known type
 * (i = integer)
 * fixed some bugs
 * this version ran without any problems, and was released as tclbin-alpha0.2
 * for more than a month. (just forgot to check it in)
 *
 * Revision 1.4  1994/11/21  21:18:41  dl
 * works ok - bin_new allows to attach to a 'father' (so its like ptr)
 * bin_move to move objects
 * version with object(_obj,xxx) - to be replaced with bin_info
 *
 * Revision 1.3  1994/11/08  23:57:24  dl
 * better error handling - new unified object trace proc calling,
 * so getobject also works with sub fields.
 *
 * Revision 1.2  1994/11/08  18:13:09  dl
 * work, added bin_read, bin_write, bin_copy, lots of fixes
 *
 * Revision 1.1  1994/10/31  23:37:17  dl
 * Initial revision
 *
 *
 */

static char *rcsid="$Id: bindata.c,v 2.15 1997/01/29 23:39:44 dl Exp $";

#include "bindata.h"
#ifdef ADONIS
#  include "complex.h"
#endif
#ifdef STDC_HEADERS
# include <string.h>
# include <ctype.h>
#endif

#include <stdio.h>

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif


	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
set doc_package_name "Tcl binary extension"

doc_title TclBin "Binary object manipulation" {
  These procedures provides access to binary objects and C structures
  from Tcl.
  
  Viewed from Tcl each C object is a Tcl array, each defined fields
  is a sub object. Trace functions provide the read/write conversions
  between Tcl strings and the Binary representation.

  See also libtclbin.ps/.dvi for an introduction and overall description.

}
# new doc type:
set _doc_t(o) "object";

# C Api infos
doc_section {C API} {

  From C, each object is a Bin_Object structure with the following fields:

  * magic (int)
    Magic number used to check integrity of the objects when the library
    has been compiled with VERIFYSTRUCT.
  * data  (void *)
    Pointer to the actual data of the object
  * size  (size_t)
    data size (used for read/write & control)
  * type (char *)
    pointer to the type name string (allocated at the end of the structure)
  * father (Bin_Object *)
    father object (if object is a sub object)
  * elemsize (size_t)
    size of a single element (if applicable : if type is an array of built
    in types)
  * DeleteProc (tfunc_obj *)
    function to be called when deleting the object (ckfree)
  * PeekPokeProc (Tcl_VarTraceProc *)
    tcl trace proc to be called for reading/writing one element of
    that type (if applicable)
  * TraceProc (Tcl_VarTraceProc *)
    tcl trace proc to be called for reading/writing that type
    (Bin_TraceObject(3) for plain objects)

  See individual man pages for Bin_NewObject(3), Bin_Def(3), ...
  for specific API informations.
}
*/

	/*---------*/

/*
 * Initialize package.
 * To be called to add the binary support to an interp (in Tcl_AppInit(3)).
 */
int
Bin_Init(interp)
     Tcl_Interp *interp; /* The current Tcl interpreter */
{
  Tcl_HashTable *Bin_TypeHashTable=(Tcl_HashTable *)
	                              ckalloc(sizeof(Tcl_HashTable));  


  Tcl_InitHashTable(Bin_TypeHashTable,TCL_STRING_KEYS);
  Tcl_CreateCommand(interp,"bin_new",Bin_NewCmd,
                    (ClientData)Bin_TypeHashTable,NULL);
  Tcl_CreateCommand(interp,"bin_def",Bin_DefCmd,
                    (ClientData)Bin_TypeHashTable,NULL);
  Tcl_CreateCommand(interp,"bin_info",Bin_InfoCmd,
		    (ClientData)NULL,NULL);
  Tcl_CreateCommand(interp,"bin_sizeof",Bin_SizeofCmd,
                    (ClientData)Bin_TypeHashTable,NULL);
  Tcl_CreateCommand(interp,"bin_move",Bin_MoveCmd,
                    (ClientData)NULL,NULL);
  Tcl_CreateCommand(interp,"bin_resize",Bin_ResizeCmd,
                    (ClientData)NULL,NULL);  
  Tcl_CreateCommand(interp,"bin_copy",Bin_CopyCmd,
                    (ClientData)NULL,NULL);
  Tcl_CreateCommand(interp,"bin_read",Bin_ReadCmd,
		    (ClientData)NULL,NULL);
  Tcl_CreateCommand(interp,"bin_write",Bin_WriteCmd,
		    (ClientData)NULL,NULL);
/*
 * Register specialised I/Os functions for each needed type :
 */ 
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "char",sizeof(char),Bin_TraceChar)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "int",sizeof(int),Bin_TraceInt)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "short",sizeof(short),Bin_TraceShort)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "netint",4,Bin_TraceNetInt)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "netshort",2,Bin_TraceNetShort)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "double",sizeof(double),Bin_TraceDouble)==TCL_ERROR) {
    return TCL_ERROR;
  }
#ifdef ADONIS
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		     "complex",sizeof(complex),Bin_TraceComplex)==TCL_ERROR) {
    return TCL_ERROR;
  }
#endif
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "float",sizeof(float),Bin_TraceFloat)==TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Bin_RegisterType(interp,Bin_TypeHashTable,
		       "str",sizeof(char *),Bin_TraceStr)==TCL_ERROR) {
    return TCL_ERROR;
  }

#if ( ( TCL_MAJOR_VERSION > 7 ) || \
      ( (TCL_MAJOR_VERSION == 7) && (TCL_MINOR_VERSION >= 5) ) )
  /* Register as a package */
  if (Tcl_PkgProvide(interp, "Bin", BIN_VERSION_STR) != TCL_OK) {
    return TCL_ERROR;
  }
#endif
  return TCL_OK;
}

	/*---------*/

/*
 * Register a new binary type
 * A new type is created for latyer use by bin_def(n)/Bin_Def(3).
 */
int
Bin_RegisterType(interp, ht, typename, size, traceProc)  
    Tcl_Interp *interp; /* Pointer to current Tcl interp structure */
    Tcl_HashTable *ht;/*
		       * Pointer to the hash table where the types are stored
		       * If NULL, Bin_GetTHT(3) is called to get the 
		       * current value.
		       */
    CONST char *typename; /* Name of the type */
    int size;	/* size of the type (used for arrays of the type) */
    Tcl_VarTraceProc *traceProc; /* conversion proc to call */
{
  Tcl_HashEntry *entryPtr;
  int new;
  Bin_TypeStruct *bts;

  if (ht==NULL) {
    if ((ht=Bin_GetTHT(interp))==NULL) return TCL_ERROR;
  }
  
  entryPtr=Tcl_CreateHashEntry(ht,(char *)typename,&new);
  if (!new)
    {
      Tcl_AppendResult(interp,"name \"",typename,"\" already allocated",
                       (char *)NULL );
      return TCL_ERROR;
    }
  bts= (Bin_TypeStruct *)ckalloc(sizeof(Bin_TypeStruct));
  if (bts==NULL) {
	Tcl_SetResult(interp,"Can't allocate type structure object!",
		      TCL_STATIC);
	return TCL_ERROR;
  }
#ifdef VERIFYSTRUCT
  bts->magic=MAGIC_TYP;
#endif
  bts->TraceProc=traceProc;
  bts->size=size;
  Tcl_SetHashValue(entryPtr,bts);
  return TCL_OK;
}

	/*---------*/

/*
 * Get types Hash Tables
 *
 * This function can be called when you want to add several new types outside
 * of Bin_Init (In the new module compatible with tclbin for instance). 
 * It will not work if bin_new(n) has been renamed.
 *
 * Returns the required hash table for registered types.
 *
 */
Tcl_HashTable *
Bin_GetTHT(interp)  
    Tcl_Interp *interp; /* Pointer to current Tcl interp structure */
{
  Tcl_CmdInfo info;
  
  if (Tcl_GetCommandInfo(interp,"bin_new",&info)==0) {
      Tcl_AppendResult(interp,"Can't retrieve Bin Type Hash Table,",
		       "\"bin_new\" has probably been renamed or",
		       "tclbin is not enabled in this Tcl interp", NULL);
      return NULL;
    }
  return (Tcl_HashTable *)(info.clientData);
}

	/*---------*/

/*
 * generate uniq Id
 * It concatenates the given base with a serial number.
 * returns an uniq id, from static memory.
 *
 * warning:
 *   Only the first 10 characters of the base name are used if it is larger.
 */
CONST char *
Bin_GenerateName(base)
    CONST char *base; /* base name to which the numeric id will be added */
{
    static int id=1;
    static char name[10+22]; /* should be enough even for 64 bits numbers */
    sprintf(name,"%.10s%d",base,id++);
    return name;
}

	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpeproc bin_new {
  o:objectname   # object name or "#auto" for automatically generated name
  s:typename     # a type name (like `char*', `int*', `double*', `mytype',...)
  i:sizeExpr     # expression giving the size in bytes of the object
  o:father  ""   # father object name (container)
  i:offsetExpr 0 # expression giving the offset in bytes from father start, '
                 # where to attach this object
  -no_unset 0    # if this argument is present, the object will not be
                 # automatically unset when father is unset (dangerous)
} {
        Create a new object named `objectname' of type `typename'.
        The object created is `sizeExpr' bytes large.
        If `father' is given, no new memory is allocated and the object
        is then just a pointer to a (sub)space of father data.

        If the type is a registered type plus a star (*), like `int*',
        the object is considered like an array of the base type. And
        accessing `objectname'(#i#) where #i# is a number will access
        the data at position #i# (starting at 0). Else, the typename
        is only an information string, that can be used to check
        the data allocated destination usage if needed.

	For all objects, besides additional fields defined using bin_def(n),
	the following fields are defined:
	* _obj_
	  Read only type name (all other fields
	  can be used either to get the object content or to modify it).
        * _str_
	  String representation of the object. Every instance of the character
          ascii value zero ('\\0') is replaced by "`\\0'" and the
	  backslash itself (\\) is replaced by "`\\\\'".
	* _hex_
	  Hexadecimal representation of the object: a string of twice
          the size of the object as an hexadecimal number.
        * #i#`,hex'
          Peek/poke a byte as a two characters hexadecimal number
          at offset #i# within the object.
	* #i#
	  Access #i#`th' element (starting at zero) of the object
          when it has been defined as an array of a built in 
	  type (`typename' is "`float*'" for instance).

	Returns the object name. 

	Memory is allocated for the new object (unless "father" is given).
} {	
bin_new tab double* 10*[bin_sizeof double]
bin_new el2 double 8 tab1 2*8
puts $tab(_obj_) ;
puts [expr 0x$tab(3,hex)+3] ;
set tab(3,hex) 2;
puts $tab(0); 
}

*/

	/*---------*/

/*
 * Create a new object
 *
 * see bin_new(n)
 *
 */
int
Bin_NewCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (Types Hash Table) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object  *objptr,*father;
  Bin_TypeStruct *bts;
  long size,offset;
  CONST char *name,*typename,*fathername;
  int lg;
  
  if (cdata==NULL) {
    Tcl_AppendResult(interp, "Called Bin_NewCmd with NULL client data",NULL);
    return TCL_ERROR;
  }

  if (argc<4 || argc>7) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	" object|#auto type sizeExpr ?father? ?offsetExpr? ?no_unset?\"",
		     (char *) NULL);
	return TCL_ERROR;
  }
  if (!strcmp(argv[1],"#auto")) 
    name=Bin_GenerateName(argv[2]); 
  else 
    name=argv[1];

  typename=argv[2];
  
  if (Tcl_ExprLong(interp,argv[3],&size)==TCL_ERROR) {return TCL_ERROR;}

  /* check object is not already defined (like in GetObject, but reversed
     error condition */
  if (_BIN_GETOBJECT(interp,name)) {
    Tcl_AppendResult(interp,"\"",name,"\" is already an object",NULL);
    return TCL_ERROR;
  }

  if (argc>4) {
    fathername= argv[4] ;
    if (!(father=Bin_GetObject(interp,fathername))) 
      return TCL_ERROR;
    if (argc>5) {
      if (Tcl_ExprLong(interp,argv[5],&offset)==TCL_ERROR) return TCL_ERROR;
    } else 
      offset=0;
    if (size+offset > father->size) {
      Tcl_AppendResult(interp,"request offset+size > father's size",NULL);
      return TCL_ERROR;
    }
    objptr=Bin_NewObject((char*)father->data+offset,size,father,
			 argv[2],Bin_TraceObject);
  } else {
    fathername= NULL;
    objptr=Bin_NewObject(NULL,size,(Bin_Object *)NULL,
			 argv[2],Bin_TraceObject);
  }

  if (objptr==NULL) {
      Tcl_SetResult(interp,"Can't allocate object!",TCL_STATIC);
      return TCL_ERROR;
  }
  /* is the type an array like xxxx*, xxxx being a 'known' type (ptr) ? */
  lg=strlen(argv[2]);
  if ( ( lg>1 )  && ( argv[2][lg-1]=='*') ) {
    char *tname=ckalloc(lg);
    strncpy(tname,argv[2],lg-1);
    tname[lg-1]=0;
    bts=(Bin_TypeStruct *)Bin_GetHashVal((Tcl_HashTable *)cdata,tname);
    ckfree(tname);
    if (bts!=NULL) {
#ifdef VERIFYSTRUCT
      if (bts->magic!=MAGIC_TYP) {
	  Tcl_AppendResult(interp, "BUG: Got a type which is not a type !",
			   NULL);
	  return TCL_ERROR;
      }
#endif
      objptr->elemsize=bts->size;
      objptr->PeekPokeProc=bts->TraceProc;
   }
  }
  return Bin_TclAttach(interp, objptr, name,
       (argc>=7) ? NULL : fathername, /* if no_unset flag: null father */
		       BIN_NO_FATHER_CHECK);
}

	/*---------*/

/*
 * Creates a new binary object.
 * C API to create a new object.
 * if the ptr argument is NULL, allocate the data part too.
 *
 * See TclBin(n) "C API" sections for Bin_Object structure description 
 * (and/or bindata.h)
 *
 * Warning: this function does the memory allocation and object structure
 * initialisation but does not make the object accessible from Tcl
 * (call Bin_TclAttach(3) for complete object creation).
 */
Bin_Object *
Bin_NewObject(ptr,size,father,typename,tproc)
  void *ptr;   /* Data part of the object */
  int  size;   /* Size of the object */
  Bin_Object *father;  /* Pointer to the father of the object, if any */
  CONST char *typename;    /* name of the type of object */
  Tcl_VarTraceProc *tproc; /* Conversion proc */
{
  Bin_Object *res;
  if (ptr==NULL) {
    /* data ptr is null, we allocate the data */
    /* the object structure and the room for the type name string */
    /* at the same time */
      /* avoid BUS ERRORS with pading : */
      int pad;
      pad=sizeof(Bin_Object)%sizeof(double);
      if (pad) pad=sizeof(double)-pad;
    res=(Bin_Object *)ckalloc(sizeof(Bin_Object)+pad+size+strlen(typename)+1);
    if (res==NULL) return NULL;
    ptr=((char*)res)+sizeof(Bin_Object)+pad;
    res->type=(char *)strcpy((char*)ptr+size,typename);
  } else {
    /* only allocate the object structure + room for the type name */
    res=(Bin_Object *)ckalloc(sizeof(Bin_Object)+strlen(typename)+1);
    if (res==NULL) return NULL;
    res->type=(char *)strcpy((char*)res+sizeof(Bin_Object),typename);
  }
#ifdef VERIFYSTRUCT
  res->magic=MAGIC_OBJ;
#endif 
  res->data=ptr;
  res->size=size;
  res->father=father;
  res->TraceProc=tproc;

  /* the caller can change those 3 later if applicable */
  res->elemsize=0;
  res->PeekPokeProc=NULL;
#ifdef TCL_MEM_DEBUG
  res->DeleteProc=(tfunc_obj*)Tcl_DbCkfree;
#else
  res->DeleteProc=(tfunc_obj*)free;
#endif
  return res;
}

/*
 * Delete Object
 *
 */
void
Bin_DeleteObject (object) 
    Bin_Object *object;  /* Pointer to the object to delete */
{
    if (object->DeleteProc) {
#ifdef TCL_MEM_DEBUG
	if (object->DeleteProc==(tfunc_obj*)Tcl_DbCkfree) {
	    ckfree((void *)object);
	} else {
#endif
	    (*object->DeleteProc)(object);
#ifdef TCL_MEM_DEBUG
	}
#endif
    }
}

	/*---------*/

/*
 * Attached a C created Binary Object to Tcl
 *
 * Makes a previously created (Bin_NewObject(3)) object visible from Tcl
 * by creating appropriate Tcl_Traces. If fathername is not NULL also
 * sets up an unset callback to delete the child when father gets unset
 * (or out of scope). 
 *
 * flags is an ORed combination of value,
 * currently only : BIN_NO_FATHER_CHECK to be used when you are sure
 * that the fathername you give is indeed the name of objptr->father.
 *
 * Upon success sets the interp's result to `name'.
 */
int
Bin_TclAttach(interp,objptr,name,fathername,flags)
    Tcl_Interp *interp;         /* Current interpreter. */
    Bin_Object *objptr;         /* object structure pointer */
    CONST char *name;		/* object name (must not contain "()") */
    CONST char *fathername;	/* father name or NULL */
    int  flags;			/* flags 0: normal or BIN_NO_FATHER_CHECK */ 
{
  Bin_Object *father;
  
#ifdef VERIFYSTRUCT
  if (objptr->magic!=MAGIC_OBJ) {
    Tcl_AppendResult(interp,"\"", name,
	    "\" has a wrong structure magic number (not an object (bug)!)",
		     NULL);
    return TCL_ERROR;
  }
#endif
  if (Tcl_SetVar2(interp,name,"_obj_",objptr->type,TCL_LEAVE_ERR_MSG)==NULL) 
    return TCL_ERROR;
  if (Tcl_TraceVar2(interp,name,NULL,
		    TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		    Bin_MainTraceProc,objptr) != TCL_OK)
    return TCL_ERROR;
  if (fathername != NULL ) {
    ClientData clientData;
    char *cname;
    
    /* This is a child object and "no_unset" was not used:
     * prepare unset callback :
     */

    /* usually recheck father : */
    if ( !(flags & BIN_NO_FATHER_CHECK) ) {
      if (!(father=Bin_GetObject(interp,fathername))) 
	return TCL_ERROR;
      if (father!=objptr->father) {
	Tcl_AppendResult(interp,"\"", fathername , 
			 "\" is not \"", name, "\"'s father (bug)", NULL);
	return TCL_ERROR;
      }
    }
    /*
     * Get advised so we can unset child if father is unset 
     * by setting an unset callback on father
     */
    /* search all callbacks so we don't put twice the same callback */
    clientData=NULL;
    while (1) {
      clientData = Tcl_VarTraceInfo(interp,fathername,0,
				    Bin_FreeChild,clientData);
      if (clientData==NULL) break; 		   /* end of callback list */
      if (!strcmp((char *)clientData,name)) break; /* already set */
    }
    if (clientData==NULL) { /* was not already set, so we add it: */
      cname=ckalloc(strlen(name)+1); /* needed (to be double checked) */
      strcpy(cname,name);
      Tcl_TraceVar(interp,fathername,TCL_TRACE_UNSETS,
		   Bin_FreeChild,cname);
      /* Ideally we should remove that unset trace when child is unset... */
    }
  }
  Tcl_SetResult(interp,name,TCL_VOLATILE);
  return TCL_OK;
}
	/*---------*/

/*
 * Get an hash table value, by its name
 *
 * returns NULL if not found
 */
ClientData
Bin_GetHashVal (ht,name)
     Tcl_HashTable *ht;   /* Pointer to the Hash Table structure */
     CONST char    *name; /* name of the searched key */
{
  Tcl_HashEntry *entryPtr;
  if ((entryPtr=Tcl_FindHashEntry(ht,name))!=NULL)
    return Tcl_GetHashValue(entryPtr);
  else
    return NULL;
}

	/*---------*/

/*
 * Easily callable from C version of Bin_DefCmd(3)/bin_def(n) (C API).
 *
 * If nelems is not 1, it is assumed that an array of nelems elements
 * each of type typename and bin_sizeof(typename) bytes is being defined,
 * and the typename for the resulting field will get postfixed by a star.
 * If an array field is likewise defined, It can not be used
 * directly from Tcl (because you can not use $obj(array)(6), helas...)
 * But can still be passed to C functions needing that type, or can be
 * used by defining a pointer object to it :
 * like for instance :
 * "bin_new ptarr double* [bin_sizeof obj(array)] obj(array);"
 *
 */
int
Bin_Def(ht,interp,name1,name2,typename,offset,nelems) 
    Tcl_HashTable *ht;    /* Type Hash Table, if NULL, 
                           * Bin_GetTHT(3) will be called to retrieve it*/
    Tcl_Interp *interp;         /* Current interpreter. */
    CONST char *name1;		/* object name */
    CONST char *name2;		/* field name */
    CONST char *typename;	/* type name */
    int offset;			/* offset within father */
    long nelems;		/* 1 or number of elements for arrays */
{
  Tcl_VarTraceProc traceP;
  Bin_TypeStruct   *bts;
  Bin_Object  *objptr,*nobj;

  if (ht==NULL) {
    if ((ht=Bin_GetTHT(interp))==NULL) return TCL_ERROR;
  }
  
  bts=(Bin_TypeStruct *)Bin_GetHashVal(ht,typename);
  if (bts==NULL) {
    Tcl_AppendResult(interp, "\"",typename,"\" is not registered type",NULL);
    return TCL_ERROR;
  }
#ifdef VERIFYSTRUCT
  if (bts->magic!=MAGIC_TYP) {
    Tcl_AppendResult(interp, "BUG: Got a type which is not a type !",NULL);
    return TCL_ERROR;
  }
#endif

  if ((objptr=Bin_GetObject(interp,name1))==NULL) return TCL_ERROR;

  if (_BIN_GETOBJECT2(interp,name1,name2)) {
    Tcl_AppendResult(interp,"field \"",name2,"\" of object \"",name1,
		"\" already defined",(char *)NULL);
    return TCL_ERROR;
  }


  if ( (offset < 0) || ( (offset+bts->size*nelems) > objptr->size) ) {
    Tcl_AppendResult(interp, "(offset+type size) goes outside of object",NULL);
    return TCL_ERROR;
  }
  
  if (nelems==1) {
    nobj=Bin_NewObject(((char*)objptr->data)+offset,bts->size,
		       objptr,typename,bts->TraceProc);
  } else {
    int lg=strlen(typename);
    char *ptname=ckalloc(lg+2);
    strcpy(ptname,typename);
    ptname[lg]='*';
    ptname[lg+1]=0;
    nobj=Bin_NewObject(((char*)objptr->data)+offset,bts->size*nelems,
		       objptr,ptname,Bin_TracePtr);
    ckfree(ptname);
  }
  if (nobj==NULL) {
    Tcl_SetResult(interp,"Can't allocate object!",TCL_STATIC);
    return TCL_ERROR;
  }
  if (nelems!=1) {
      nobj->elemsize=bts->size;
      nobj->PeekPokeProc=bts->TraceProc;
  }
  Tcl_TraceVar2(interp,name1,name2,
		TCL_TRACE_READS|TCL_TRACE_WRITES|TCL_TRACE_UNSETS,
		Bin_MainTraceProc,nobj);
  return TCL_OK;
}
 
	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpeproc bin_def {
  o:object	# base object name (array name)
  s:field	# field name
  s:type	# registered type name (built in)
  i:offsetExpr  # offset in bytes from base object first byte
  i:nelems 0    # number of elements (to create an array/pointer)
} {
  Define a field named within object.
  Field `field' is defined inside the binary object `object' 
  (like C struct) and the corresponding TCL associative array field
  is attached. If you use give the optional `nelems` argument,
  a field of type `type*' is created: an array of Nelems elements of
  type `type'. Due to limitation in Tcl array handling you can't
  use obj(array)(n) directly from Tcl but you can through a 
  child/pointer/sub object or as a special case, for `char*' like `str'.

  TclBin provides the following built in types : 
  * char
  character (if used with nelems option, makes a fixed length
  string)
  * int
  integer (machine dependant, same as C type)
  * short
  short integer (machine dependant, same as C type)
  * netint
  32 bits unsigned integer number, in standard network order (high
  byte first)
  * netshort
  16 bits unsigned integer number, in standard network order.
  * double
  double precision floating point number (machine dependant, same as C
  type (in general IEEE)).
  * float
  simple precision floating point number (same as C).
  * str
  Null terminated, variable length (but not going outside object), 
  array of characters. Use char arrays instead !

  It returns the string "`object(field)'".
  
} {
bin_new toto double* [bin_sizeof double]*10
bin_def toto dbl double 0
set toto(dbl) 5.6
}
*/

	/*---------*/

/*
 * Get the object from its name.
 * Search the interpretor for the object by its name.
 * Returns the pointer to the object structure or NULL if the object
 * is not found. In that case an error message is set in the Tcl
 * interp.
 *
 * Internals:
 *  The object is searched by looking for the Bin_MainTraceProc(3)
 *  trace procedure (calling Tcl_VarTraceInfo(3)) attached 
 *  to the object name (by Bin_TclAttach(3) and Bin_Def(3)) at creation.
 */
Bin_Object *
Bin_GetObject(interp,name) 
    Tcl_Interp *interp;			/* Current interpreter. */
    CONST char *name;   /* name of the object searched */
{
    Bin_Object *res;
    if ((res=_BIN_GETOBJECT(interp,(char *)name))==NULL) {
        Tcl_AppendResult(interp,"\"", name,"\" is not an object",NULL);
	return NULL;
    }
#ifdef VERIFYSTRUCT
    if (res->magic!=MAGIC_OBJ) {
	Tcl_AppendResult(interp,"\"", name,
	  "\" has a wrong structure magic number (not an object (bug)!)",
	  NULL);
	return NULL;
    }
#endif
    return res;
}

	/*---------*/

/*
 * Get the object & check type.
 * Like Bin_GetObject(3) but in addition checks the type of the object
 * and returns NULL if not matching, setting the interp error message.
 */
Bin_Object *
Bin_GetObjectAndCheck(interp,name,type) 
    Tcl_Interp *interp;			/* Current interpreter. */
    CONST char *name;    /* name of the object to search */
    CONST char *type;	/* type to check against object's */
{
    Bin_Object *res;

    if (!(res=Bin_GetObject(interp,name))) return NULL;
    if (strcmp(res->type,type)) {
	Tcl_AppendResult(interp,"\"", name,"\" is of type ", res->type,
		" and not of expected type ",type, (char *) NULL);
	return NULL;
    }
    return res;
}

	/*---------*/


/* Master entry point of all callbacks
 *
 * This trace is attached to every TclBin objects, sub objects, fields
 * and calls the appropriate trace proc depending of the kind of
 * object.
 *
 * See Bin_GetObject(3) "Internals" section for more infos.
 */
char *
Bin_MainTraceProc (cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  
  if (object==NULL) {
    return "Null pointer for object data in trace!";
  }
#ifdef VERIFYSTRUCT
  if (object->magic!=MAGIC_OBJ) {
    return "BUG: Data part of trace is not an object !";
  }
#endif
#ifdef DEBUG
  fprintf(stderr,"called trace for %s(%s) - f=%d\n",name1,
	  name2 ? name2 : "",
	  flags);
  fprintf(stderr,"\tdata=%p,objtype=%s,size=%d, calling %p\n",
	object->data,object->type,object->size,object->TraceProc);
#endif
  if (object->TraceProc)
	return (*(object->TraceProc))(cdata,interp,name1,name2,flags);
  else
	return "NULL TraceProc for object!";
}


	/*---------*/


/*
 * Main Trace each object, allows tcl <-> whole object access, and
 * hex peek/poke
 */
char *
Bin_TraceObject(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  int n;
  char *str;

  if (flags & TCL_TRACE_READS)  {
    /*
     * READ Operations :
     */
    /* short cut : */
    if (name2==NULL || (*name2==0) || (*name2!='_' && !isdigit(*name2))) 
      return NULL;
    n=strtol(name2,&str,10);
    if (*str==0) {
	if (object->PeekPokeProc) {
	    Bin_Object *oi;
	    char *res;
	    if ((n+1)*object->elemsize>object->size) return "out of bounds";
	    oi=Bin_NewObject((char*)object->data+n*object->elemsize,
			     object->elemsize,
			     object,
			     "peek-poke",
			     object->PeekPokeProc);
	    if (oi==NULL) return "can't alloc peek-poke object";
	    res=(Bin_MainTraceProc)(oi,interp,name1,name2,flags);
	    Bin_DeleteObject(oi);
	    return res;
	} else {
	    return "illegal use of $array(n) syntax for non pointer object";
	}
    } else if ( str!=name2 && !strcmp(str,",hex") ) {
      /*
       * Peek hex value at a given offset :
       */
      static char res[3]={0,0,0};  /* for 2 bytes string : hex val of a byte*/
      if (n<0 || n>=object->size) return "offset out of range";
      sprintf(res,"%02x", *((unsigned char *)object->data + n));
      Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
    } else if (!strcmp(name2,"_hex_")) {
      /*
       * Hexadecimal output (whole object data) :
       */
      char *bufs,*pres,c;
      unsigned char *ps,v;
	    
      bufs=ckalloc(2*(object->size)+1);
      if (bufs==NULL) return "can't alloc memory for result";
      for (n=object->size,ps=(unsigned char*)object->data,pres=bufs;n--;) {
	v=*ps++;
	c= (v>>4);
	*pres++= (c<=9)?c+'0':c+'a'-10;
	c= v&0xf;
	*pres++= (c<=9)?c+'0':c+'a'-10;
      }
      *pres=0;
      Tcl_SetVar2(interp,name1,name2,bufs, flags&TCL_GLOBAL_ONLY);
      ckfree(bufs);
    } else if (!strcmp(name2,"_str_")) {
      /*
       * String output (whole object data, 0 escaped to \0 and \ into \\) :
       */
      Tcl_DString res;
      char *ps;
	    
      Tcl_DStringInit(&res);
      for (n=object->size,ps=object->data;n--;ps++) {
	switch (*ps) {
	case 0 :
	  Tcl_DStringAppend(&res,"\\0",2);
	  break;
	case '\\' :
	  Tcl_DStringAppend(&res,"\\\\",2);
	  break;
	default :
	  Tcl_DStringAppend(&res,ps,1);
	  break;
	}
      }
      Tcl_SetVar2(interp,name1,name2,Tcl_DStringValue(&res),
		  flags&TCL_GLOBAL_ONLY);
      Tcl_DStringFree(&res);
    }
  } else if (flags & TCL_TRACE_WRITES) {
    /*
     * WRITE Operations :
     */
    /* short cut */
    if (name2==NULL || (*name2==0) || (*name2!='_' && !isdigit(*name2))) 
      return NULL; 
    n=strtol(name2,&str,10);
    if (*str==0) {
	if (object->PeekPokeProc) {
	    Bin_Object *oi;
	    char *res;
	    if ((n+1)*object->elemsize>object->size) return "out of bounds";
	    oi=Bin_NewObject((char*)object->data+n*object->elemsize,
			     object->elemsize,
			     object,
			     "peek-poke",
			     object->PeekPokeProc);
	    if (oi==NULL) return "can't alloc peek-poke object";
	    res=(Bin_MainTraceProc)(oi,interp,name1,name2,flags);
	    Bin_DeleteObject(oi);
	    return res;
	} else {
	    return "illegal use of set array(n) syntax for non pointer object";
	}
    } else if ( str!=name2 && !strcmp(str,",hex") ) {
	/*
	 * Poke hex value at a given offset :
	 */
	unsigned char v=0;
	char *r;
	if (n<0 || n>=object->size) return "offset out of range";
	r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
	if (strlen(r)!=2
#ifdef VERIFYHEX
	    || !strchr("0123456789abcdef",r[0])
	    || !strchr("0123456789abcdef",r[1])
#endif
	    )
	    return "not a valid hex number";
	v=  ( (r[0]<='9')?(r[0]-'0'):(r[0]-'a'+10) ) << 4;
	v+= ( (r[1]<='9')?(r[1]-'0'):(r[1]-'a'+10) ) ;
#ifdef DEBUG
	fprintf(stderr,"setting, read '%s', will set : %x\n",r,(int)v);
#endif
	*((unsigned char *)object->data + n)=v;
    }
    else if (!strcmp(name2,"_hex_")) {
	/*
	 * Hexadecimal input (whole object data) :
	 */
	char *r,c;
	unsigned char *po,v;
	
	r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
	if (strlen(r)!=2*(object->size)) return "incorrect string length";
	for (n=object->size,po=(unsigned char*)object->data;n--;) {
	    c=*r++;
#ifdef VERIFYHEX
	    if (!strchr("0123456789abcdef",c)) return "not a valid hex string";
#endif
	    v = ( (c<='9')?(c-'0'):(c-'a'+10) ) << 4;
	    c=*r++;
#ifdef VERIFYHEX
	    if (!strchr("0123456789abcdef",c)) return "not a valid hex string";
#endif
	    v += ( (c<='9')?(c-'0'):(c-'a'+10) );
	    *po++ = v;
	}
    } else if (!strcmp(name2,"_str_")) {
	/*
	 * String input (whole object data, 0 escaped to \0 and \ into \\) :
	 */
	int s;
	char c,*r,*po;
	    
      r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
      po=object->data; n=0; s=object->size;
      while ((c=*r++)) {
	n++;
	if (n>s) return "string too large for object";
	if (c=='\\') {
	  if (!*r) return "\\ at the end of the string";
	  switch (*r++) {
	  case '0' :
	    *po++=0;
	    break;
	  case '\\':
	    *po++='\\';
	    break;
	  default :
	    return "unexpected \\ sequence";
	    break;
	  }
	} else {
	  *po++=c;
	}
      }
      if (n!=s) return "string too short for object";
    } else if (!strcmp(name2,"_obj_")) {
	/* make _obj_ readonly (type) */
	Tcl_SetVar2(interp,name1,name2,object->type,0);
	return "object field is read only";
    }
  } else if (flags&TCL_TRACE_DESTROYED) {
    /*
     * UNSET Operation :
     */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
	
  return NULL;    
}

	/*---------*/


/*
 * get the size of a registered type
 * C api to get check existence and return the type->size of a 
 * type given its name.
 * returns the size of a type or -1 if the type is not found (in which case
 * an error message is optionally set in the Tcl interp)
 */
int
Bin_SizeofType(ht,interp,typename) 
    Tcl_HashTable *ht;    /* Type Hash Table, if NULL, 
                           * Bin_GetTHT(3) will be called to retrieve it*/
    Tcl_Interp *interp;   /* Current interpreter, if NULL no error
			   * will be set if type is not found.
			   * (NB: ht must be given when interp is not)
			   */
    CONST char *typename; /* type name */
{
  Bin_TypeStruct   *bts;
  if (ht==NULL) {
    if (interp==NULL) return -1; /* caller is wrong */
    if ((ht=Bin_GetTHT(interp))==NULL) return -1;/* types ht not found ! */
  }
  
  bts=(Bin_TypeStruct *)Bin_GetHashVal( ht , typename) ;
  if (bts==NULL) {
    if (interp) Tcl_AppendResult(interp,"\"", typename ,
				 "\" is not a registered type", NULL);
    return -1;
  } else {
#ifdef VERIFYSTRUCT
    if (bts->magic!=MAGIC_TYP) {
      if (interp) Tcl_AppendResult(interp, 
				   "BUG: Got a type which is not a type !",
				   NULL);
      return -1;
    }
#endif
    return bts->size;
  }  	
}

	/*---------*/


#ifndef __C2MAN__ /* no doc needed for commands called by Tcl interp
                   * or used internally (builtin types TraceProcs)
                   */

	/*---------*/

/*
 * Associate a field with a memory location
 *
 * usage : see bin_def(n)
 */
int
Bin_DefCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (Types Hash Table) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  long offset,nelems;

  if (argc<5 || argc>6) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
            " object field type offsetExpr ?nelems?\"", (char *) NULL);
    return TCL_ERROR;
  }
  if (cdata==NULL) {
    Tcl_AppendResult(interp, "Called Bin_DefCmd with NULL client data",NULL);
    return TCL_ERROR;
  }

  if (Tcl_ExprLong(interp,argv[4],&offset)==TCL_ERROR)
    return TCL_ERROR;
  if (argc>5) {
    if (Tcl_ExprLong(interp,argv[5],&nelems)==TCL_ERROR)
      return TCL_ERROR;
    if (nelems<1) {
      Tcl_AppendResult(interp, "number of elements in array must be > 0",
		       (char *) NULL);
      return TCL_ERROR;
    }
  } else
    nelems=1;
  if (Bin_Def((Tcl_HashTable *)cdata,
	      interp,argv[1],argv[2],argv[3],offset,nelems)==TCL_ERROR)
      return TCL_ERROR;
  Tcl_AppendResult(interp,argv[1],"(",argv[2],")",NULL);
  return TCL_OK;
}

	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpproc bin_sizeof {
  type|object   # name of object or type which size is wanted
} {
 returns the size in bytes of a type or object.
}
*/
	/*---------*/

/*
 * return the sizeof(type) or object
 *
 * usage : see bin_sizeof(n)
 */
int
Bin_SizeofCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (Types Hash Table) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object  *objptr;
  int size;
  

  if (argc!=2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		     " type|object\"", (char *) NULL);
    return TCL_ERROR;
  }
  if (cdata==NULL) {
    Tcl_AppendResult(interp, "Called Bin_SizeofCmd with NULL client data",
		     NULL);
    return TCL_ERROR;
  }
  /* we pass NULL as interp because Bin_GetObject will set the start of
     the not found error message */
  if ((size=Bin_SizeofType((Tcl_HashTable *)cdata, NULL, argv[1]))==-1) {
    if ((objptr=Bin_GetObject(interp,argv[1]))==NULL) {
      Tcl_AppendResult(interp, " nor a registered type",NULL);
      return TCL_ERROR;
    }
    size=objptr->size;
  }
  Tcl_SetResult(interp,ltoa(size),TCL_STATIC);
  return TCL_OK;
}

	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpproc bin_info {
  -type 0 # If the `-type' flag is used, only the object's type is returned.
  o:object   # name of object
} {
 Dumps the Bin_Object structure of `object' as a keyed list, or returns
 the object's type.
 The following keys with values are returned:

   * data  
   Memory data pointer address (should not be used beside debugging)
   * size
   Size in bytes (given more easily by bin_sizeof(n))
   * father
   Father object pointer address (or NULL)
   * type
   Type name. Also given by the `_obj_' field (which is not available
   for fields, so this is currently the only normal use of this
   command, beside debugging: to get the object type).
   * elemsize
   Size of one element (in case of arrays)
   * deleteproc, peekpokeproc, traceproc
    Internal callbacks used in object management (see "C API" section)
}
*/
	/*---------*/

/*
 * return infos about an object
 *
 * usage : see bin_info(n)
 *
 */
int
Bin_InfoCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object  *object;
  char buf[24]; /* enough for a long/ptr */
  char *cname=argv[0];
  int  typeonly=0;

  if (argc>1 && (strcmp(argv[1], "-type") == 0) ) {
    typeonly=1;
    argv++;
    argc--;
  }
  if (argc!=2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", cname,
                       " ?-type? object\"", (char *) NULL);
	return TCL_ERROR;
  }
  if ((object=Bin_GetObject(interp,argv[1]))==NULL) {
      return TCL_ERROR;
  }

if (typeonly) {
  Tcl_SetResult(interp,object->type,TCL_VOLATILE);
} else {
  sprintf(buf,"%p",object->data);
  Tcl_AppendResult(interp,"{data ",buf,"} ",NULL);
  sprintf(buf,"%d",object->size);
  Tcl_AppendResult(interp,"{size ",buf,"} ",NULL);
  sprintf(buf,"%p",object->father);
  Tcl_AppendResult(interp,"{father ",buf,"} ",NULL);
  /* to be fixed because object->type could include silly { },... */
  Tcl_AppendResult(interp,"{type \"",object->type,"\"} ",NULL);
  sprintf(buf,"%d",object->elemsize);
  Tcl_AppendResult(interp,"{elemsize ",buf,"} ",NULL);
  sprintf(buf,"%p",object->DeleteProc);
  Tcl_AppendResult(interp,"{deleteproc ",buf,"} ",NULL);
  sprintf(buf,"%p",object->PeekPokeProc);
  Tcl_AppendResult(interp,"{peekpokeproc ",buf,"} ",NULL);
  sprintf(buf,"%p",object->TraceProc);
  Tcl_AppendResult(interp,"{traceproc ",buf,"}",NULL);
}
  return TCL_OK;
}


	/*---------*/


/*
# For extraction by AutoDoc's c2tcldoc / document
dpeproc bin_move {
  -absolute 0    # If the `-absolute' flag is used, the offset expression
                 # is absolute
  -noerr 0       # If the `-noerr' flag is used, trying to move the
                 # object outside father limits will not generate an error.
  o:object       # object name
  i:offsetExpr 0 # offset expression
  i:sizeExpr   0 # size expression

} {
        Move (incr by 1 object position, by default) a pointer object or a
        field.
        The new position, in bytes, is : `pos+sizeof(object)' by default,
        `pos+offsetExpr*sizeof(object)' when `offsetExpr' is given but
	`sizeExpr' is not given,
        or `pos+offsetExpr*sizeExpr' if both `offsetExpr' and
        `sizeExpr' are given. `pos' being the current position or father
        start when `-absolute' flag is used.

	Returns the resulting absolute offset within father.
} {
bin_move el2
bin_move -absolute el2 2 1
}
*/

	/*---------*/

/*
 * Move object pointer within father
 *
 * usage : see bin_move(n)
 *
 */
int
Bin_MoveCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object  *object,*father;
  int absolute,noerr;
  long size,offset,newoffset;
  char *cname=argv[0];

  absolute = 0;
  if ((argc > 1) && (strcmp(argv[1], "-absolute") == 0)) {
      absolute = 1;
      argv++;
      argc--;
  }

  noerr = 0;
  if ((argc > 1) && (strcmp(argv[1], "-noerr") == 0)) {
      noerr = 1;
      argv++;
      argc--;
  }

  if ( ( argc  < 2 ) ||  ( argc > 4 ) ) {
    Tcl_AppendResult(interp, "wrong # args: should be \"", cname,
		     " ?-absolute? ?-noerr? object ?offsetExpr? ?sizeExpr?\"",
		     (char *) NULL);
      return TCL_ERROR;
  }
  
  if (!(object=Bin_GetObject(interp,argv[1]))) return TCL_ERROR;
  if ((father=object->father)==NULL) {
	Tcl_AppendResult(interp, "\"", argv[1],
		"\" is not movable (not included into a container)",
			 (char *) NULL);
    return TCL_ERROR;
  }
  if (argc>2) {
      if (Tcl_ExprLong(interp,argv[2],&offset)==TCL_ERROR) {return TCL_ERROR;}
  } else offset=1;
  if (argc>3) {
      if (Tcl_ExprLong(interp,argv[3],&size)==TCL_ERROR) {return TCL_ERROR;}
  } else size=object->size;
  
  newoffset = ( absolute ? 0 
	                 : (char*)object->data-(char*)father->data 
              ) + offset*size;
  if ( newoffset < 0 ) {
      if (noerr) return TCL_OK;
      Tcl_AppendResult(interp, "trying to move \"", argv[1],
		       "\" before its father origin",
		       (char *) NULL);
      return TCL_ERROR;
  }
  if ( newoffset+object->size > father->size ) {
      if (noerr) return TCL_OK;
      Tcl_AppendResult(interp, "trying to move \"", argv[1],
		       "\" after its father end",
		       (char *) NULL);
      return TCL_ERROR;
  }
  object->data=(char*)father->data+newoffset;

  sprintf(interp->result,"%ld",newoffset);
  return TCL_OK;
}

	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpeproc bin_resize {
  o:object         # object name
  i:newSizeExpr    # size expression
} {
 Changes the size field of an object to `newSizeExpr'. Object must be
 a pointer object.

 Returns the new size.
} {
bin_new buf  buffer 128
bin_new line char*  16 buffer;
...
bin_resize line 32 
}
*/
	/*---------*/

/*
 * Resize object pointer . It must remain inside his father .
 *
 * usage : see bin_resize(n)
 *
 */
int
Bin_ResizeCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object  *object,*father;
  long size;

  if (argc != 3) {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " object sizeExpr\"",
		       (char *) NULL);
      return TCL_ERROR;
  }
  
  if (!(object=Bin_GetObject(interp,argv[1]))) return TCL_ERROR;
  if ((father=object->father)==NULL) {
    Tcl_AppendResult(interp, "\"", argv[1],
		     "\" is not resizable (not included into a container)",
		     (char *) NULL);
    return TCL_ERROR;
  }

  if (Tcl_ExprLong(interp,argv[2],&size)==TCL_ERROR) return TCL_ERROR;

  if (   ( size < 0 ) ||
      ( ((char*)object->data + size) > ((char*)father->data + father->size) )
     ) {
    Tcl_AppendResult(interp, "trying to resize \"", argv[1],
		     "\" out of father bounds",
		     (char *) NULL);
    return TCL_ERROR;
  }
  
  object->size = size ;
  
  sprintf(interp->result,"%ld",(long)object->size);
  return TCL_OK;
}


	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpproc bin_read {
  -nobuf 0   # If present will use unbuffered (read(2)) call instead
             # of stdio (fread(3)) call. For Tcl versions 7.5 or above
             # this flags does not exist as only Tcl_Read(3) can be used.
  fileId     # Any tcl file descriptor (Channel) open for reading.
  o:object   # object name
} {
  Reads "`object'"'s size bytes from `fileId' into the object.

  Returns the number of bytes actually read (can be less than object's
  size if channel configured non blocking and writing more would block
  for instance).
}

dpproc bin_write {
  -nobuf 0   # If present will use unbuffered (write(2)) call instead
             # of stdio (fwrite(3)) call. For Tcl versions 7.5 or above
             # Tcl_Write(3) is always used but Tcl_Flush(3) is called
             # if this flags is present.
  fileId     # Any tcl file descriptor (Channel) open for writing.
  o:object   # object name
} {
  Writes "`object'"'s size bytes from `object' to `fileId'.

  Returns the number of bytes actually written.
}
*/

	/*---------*/


#if ( (TCL_MAJOR_VERSION > 7) || \
      ( (TCL_MAJOR_VERSION == 7) && (TCL_MINOR_VERSION >= 5) ) )

	/*---------*/

	/* For Tcl versions since 7.5 */

	/*---------*/

/*
 * read binary data
 * usage: see bin_read(n)
 */
int
Bin_ReadCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Tcl_Channel chan;
  Bin_Object *object;
  long bytesRead;
  int i=1,mode;
   
  /* nobuf has no more a meaning in tcl7.5 */
  if ( (argc>1) && (*(argv[1])=='-') ) { /* file id can't start with '-' */
     Tcl_AppendResult(interp, "unsupported option -nobuf for \"", argv[0],
		      "\" in tcl version 7.5 or higher", (char *) NULL);
     return TCL_ERROR;
  }
  if (argc!=3) {
     Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		      " fileId object\"", (char *) NULL);
     return TCL_ERROR;
  }
  if (!(chan=Tcl_GetChannel(interp, argv[i++], &mode))) {
    return TCL_ERROR;
  }
  if (!(mode&TCL_READABLE)) {
    Tcl_AppendResult(interp, "\"", argv[i-1],
		     "\" is not readable", (char *) NULL);
    return TCL_ERROR;
  }

  if (!(object=Bin_GetObject(interp,argv[i]))) return TCL_ERROR;


  bytesRead=Tcl_Read(chan,object->data,object->size);
  if (bytesRead==-1) {
    Tcl_AppendResult(interp, "error reading \"", argv[i-1],
		     "\" : ",Tcl_PosixError(interp), (char *) NULL);
    return TCL_ERROR;
  }

  sprintf(interp->result,"%ld",bytesRead);
  return TCL_OK;
}

	/*---------*/

/*
 * write binary data
 * usage: see bin_write(n)
 */
int
Bin_WriteCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Tcl_Channel chan;
  Bin_Object *object;
  long bytesWritten;
  int nobuf=0,i=1,mode;
  char *cname=argv[0];
   
  if ( (argc>1) && (*(argv[1])=='-') ) { /* file id can't start with '-' */
     nobuf=1;        /* so any first arg with a leading '-' means -nobuf */
     argc--;
     i++;
  }
  if (argc!=3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", cname,
		" ?-nobuf? fileId object\"", (char *) NULL);
    return TCL_ERROR;
  }
  if (!(chan=Tcl_GetChannel(interp, argv[i++], &mode))) {
    return TCL_ERROR;
  }
  if (!(mode&TCL_WRITABLE)) {
    Tcl_AppendResult(interp, "\"", argv[i-1],
		     "\" is not writable", (char *) NULL);
    return TCL_ERROR;
  }
  if (!(object=Bin_GetObject(interp,argv[i]))) return TCL_ERROR;

  bytesWritten=Tcl_Write(chan,object->data,object->size);
  if (bytesWritten==-1) {
    Tcl_AppendResult(interp, "error writing \"", argv[i-1],
		     "\" : ",Tcl_PosixError(interp), (char *) NULL);
    return TCL_ERROR;
  }

  if (nobuf) {
    if (Tcl_Flush(chan)==TCL_ERROR) {
      Tcl_AppendResult(interp, "error flushing \"", argv[i-1],
		       "\" byte ",ltoa(bytesWritten)," : ",
		       Tcl_PosixError(interp), (char *) NULL);
      return TCL_ERROR;
    }
  }
  sprintf(interp->result,"%ld",bytesWritten);
  return TCL_OK;
}

	/*---------*/

#else	/* For Tcl versions before 7.5 */

	/*---------*/

/*
 * read binary data
 * usage: see bin_read(n)
 */
int
Bin_ReadCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  FILE *f;
  Bin_Object *object;
  long bytesRead;
  int nobuf=0,i=1;
  char *cname=argv[0];

  if ( (argc>1) && (*(argv[1])=='-') ) { /* file id can't start with '-' */
     nobuf=1;        /* so any first arg with a leading '-' means -nobuf */
     argc--;
     i++;
  }
  if (argc!=3) {
     Tcl_AppendResult(interp, "wrong # args: should be \"", cname,
		      " ?-nobuf? fileId object\"", (char *) NULL);
     return TCL_ERROR;
  }
  if (Tcl_GetOpenFile(interp, argv[i++], 0, 1, &f) != TCL_OK) {
    return TCL_ERROR;
  }

  if (!(object=Bin_GetObject(interp,argv[i]))) return TCL_ERROR;

  if (nobuf) {
     bytesRead = read(fileno(f),object->data, object->size);
     if (bytesRead==-1) {
	Tcl_AppendResult(interp, "error reading \"", argv[i-1],
			 "\" : ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
     }
  } else {
     clearerr(f);
     bytesRead = fread(object->data, 1, object->size, f);
     if (ferror(f)) {
	Tcl_AppendResult(interp, "error reading \"", argv[i-1],
			 "\" byte ",ltoa(bytesRead)," : ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
     } 
  }
  sprintf(interp->result,"%ld",bytesRead);
  return TCL_OK;
}

	/*---------*/

/*
 * write binary data
 * usage: see bin_write(n)
 */
int
Bin_WriteCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  FILE *f;
  Bin_Object *object;
  long bytesWritten;
  int nobuf=0,i=1;
  char *cname=argv[0];

  if ( (argc>1) && (*(argv[1])=='-') ) { /* file id can't start with '-' */
     nobuf=1;        /* so any first arg with a leading '-' means -nobuf */
     argc--;
     i++;
  }
  if (argc!=3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", cname,
		" ?-nobuf? fileId object\"", (char *) NULL);
    return TCL_ERROR;
  }
  if (Tcl_GetOpenFile(interp, argv[i++], 1, 1, &f) != TCL_OK) {
    return TCL_ERROR;
  }

  if (!(object=Bin_GetObject(interp,argv[i]))) return TCL_ERROR;

  if (nobuf) {
     bytesWritten = write(fileno(f),object->data, object->size);
     if (bytesWritten==-1) {
	Tcl_AppendResult(interp, "error writing \"", argv[i-1],
			 "\" : ",Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
     }
  } else {
     clearerr(f);
     bytesWritten = fwrite(object->data, 1, object->size, f);
     if (ferror(f)) {
	Tcl_AppendResult(interp, "error writing \"", argv[i-1],
			 "\" byte ",ltoa(bytesWritten)," : ",
			 Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
     } 
  }
  sprintf(interp->result,"%ld",bytesWritten);
  return TCL_OK;
}
	/*---------*/

#endif	/* end of tcl version selection section for bin_read/write */

	/*---------*/

/*
# For extraction by AutoDoc's c2tcldoc / document
dpproc bin_copy {
  -swapb|-rev4 0	# swap bytes if `-swapb' is given, reverse 4
                        # bytes order (1 2 3 4 to 4 3 2 1) if `-rev4'
  o:objectDest		# destination
  o:objectSrc|byteConst	# source object or byte constant
  i:offsetDestExpr 0	# offset within destination
  i:offsetSrcExpr  0	# offset within source (ignored for constant)
  i:sizeExpr	   0	# number of bytes copied
} {
  Copy (like memmove(3) (so areas can overlap)) memory from an object
  (source) to another one (destination).
  Checks source and destination size and copy maximum with these
  constraints, or the size specified by `sizeExpr'.
  If `offsetDestExpr' or `offsetSrcExpr' the copy starts with the
  given offset or else from objects.

  If source is not an object but rather an integer number, this byte
  value is then copied over the destination object (memset(3) like).

  Returns the number of bytes actually copied.
}
*/
	/*---------*/

/*
 * bin_copy copy binary data
 */
int
Bin_CopyCmd(cdata, interp, argc, argv)
    ClientData cdata;                   /* Client Data (NULL) */
    Tcl_Interp *interp;                 /* Current interpreter. */
    int argc;                           /* Number of arguments. */
    char **argv;                        /* Argument strings. */
{
  Bin_Object *odst,*osrc;
  long i,size,offD,offS;
  char *name=argv[0];
  int swapb=0,rev4=0,memsetflag=0;
  char v[4],msetv='\0';
   
  if ( (argc>1) && (argv[1][0]=='-') ) {
     rev4  =  (argv[1][1]=='r');
     swapb = !rev4;
     argc--; argv++;
  }
  if (argc<3 || argc>6) {
     Tcl_AppendResult(interp, "wrong # args: should be \"", name,
		      " ?-swapb|-rev4? objectDest objectSrc|byteConst",
		      " ?offsetDestExpr? ?offsetSrcExpr? ?sizeExpr?\"",
		      (char *) NULL);
     return TCL_ERROR;
  }
  if (!(odst=Bin_GetObject(interp,argv[1]))) return TCL_ERROR;
  name=argv[2];
  if (!(osrc=Bin_GetObject(interp,name))) {
    int v;
    if (Tcl_GetInt(interp,name,&v)==TCL_ERROR) {
      Tcl_ResetResult(interp);
      Tcl_AppendResult(interp, "\"",name, 
		       "\" is not an object nor a valid integer",NULL);  
      return TCL_ERROR;
    } /* else : */
    Tcl_ResetResult(interp);
    memsetflag=1;
    msetv=v;
  }
  if (argc>3) {
   if ( Tcl_ExprLong(interp,argv[3],&offD)==TCL_ERROR ) 
    return TCL_ERROR;
  } else offD=0;
  if (argc>4 && !memsetflag) {
   if (Tcl_ExprLong(interp,argv[4],&offS)==TCL_ERROR ) 
    return TCL_ERROR;
  } else offS=0;
  if (memsetflag)
    i=odst->size-offD;
  else
    i=MIN(osrc->size-offS,odst->size-offD);
  if (i<1) {
    Tcl_SetResult(interp,"offset larger than object size",TCL_STATIC);
    return TCL_ERROR;
  }
  if (argc>5) {
   if ( Tcl_ExprLong(interp,argv[5],&size)==TCL_ERROR ) 
    return TCL_ERROR;
   if (size>i) {
     Tcl_SetResult(interp,"size larger than object size - offset",TCL_STATIC);
     return TCL_ERROR;
   }
  } else size=i;
if (memsetflag) {
  memset((char *)odst->data+offD,msetv,size);
} else {
  if (rev4 && (size%4!=0)) {
    Tcl_SetResult(interp,"size must be multiple of 4 when using -rev4",
		  TCL_STATIC);
    return TCL_ERROR;
   }
  if (swapb && (size%2!=0)) {
      Tcl_SetResult(interp,"size must be even when using -swapb",TCL_STATIC);
      return TCL_ERROR;
   }
#define ASWAP(i,j) c=v[i];v[i]=v[j];v[j]=c
  if (swapb) {
     /* We don't use swab because swab uses pointer alignment (odd/even) */
     register char *ps=(char *)osrc->data+offS,*pd=(char *)odst->data+offD,c;
     register long n=size/2;
     for (;n--;ps+=2,pd+=2) {
	memcpy(v,ps,2);
	ASWAP(0,1);
	memcpy(pd,v,2);
     }
  } else if (rev4) {
     register char *ps=(char *)osrc->data+offS,*pd=(char *)odst->data+offD,c;
     register long n=size/4;
     for (;n--;ps+=4,pd+=4) {
	memcpy(v,ps,4);
	ASWAP(0,3);
	ASWAP(1,2);
	memcpy(pd,v,4);
     }
  } else {
     /* using memmove so areas can overlap (sub pointers for instance) */
     memmove((char *)odst->data+offD,(char *)osrc->data+offS,size);
  }
}
  sprintf(interp->result,"%ld",size);
  return TCL_OK;
}



	/*---------*/



/* I/O Char Trace */
char *
Bin_TraceChar(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[2]={0,0};

  if (flags & TCL_TRACE_READS) {
    /* Read a char : */
    *res=*((char*)object->data);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a char : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in char write";
    if (strlen(r)>1) return "bad char";
    *((char*)object->data)=*r;
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}

	/*---------*/


/* I/O Int Trace */
char *
Bin_TraceInt(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  int v;

  if (flags & TCL_TRACE_READS) {
    /* Read an int : */
    memcpy(&v,object->data,sizeof(v));  /* avoid bus error for misalignment */
    sprintf(res,"%d",v);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write an int : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in int write";
    if (Tcl_GetInt(interp,r,&v)==TCL_ERROR) return "not a valid int";
    memcpy(object->data,&v,sizeof(v));	/* avoid bus error for misalignment */
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}

	/*---------*/


/* I/O Network order Int Trace */
/* network int are always 4 bytes (32 bits) high byte first */
char *
Bin_TraceNetInt(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  
  unsigned int v;
  unsigned char *c = (unsigned char *)object->data ;
  
  if (flags & TCL_TRACE_READS) {
    /* Read a network int : */
    v = ( (*c)<<24 ) + ( (*(c+1))<<16 ) + ( (*(c+2))<<8 )  + ( *(c+3) ) ;
    sprintf(res,"%u",v);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a network int : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in netint write";
    if (Tcl_GetInt(interp,r,(int *)&v)==TCL_ERROR) return "not a valid int";
    /* in case we are running on a 64 bits ints host */
    if ( v > 0xffffffff ) return "overflow (number too large for a netint)";
    *c++ = (unsigned char) ( v >> 24 );
    *c++ = (unsigned char) ( (v & 0xff0000) >> 16 );
    *c++ = (unsigned char) ( (v & 0xff00)   >> 8  );
    *c   = (unsigned char) ( v & 0xff );
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}


	/*---------*/


/* I/O Network order Short Trace */
/* network short are always 2 bytes (16 bits) high byte first */
char *
Bin_TraceNetShort(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  
  unsigned int v;
  unsigned char *c = (unsigned char *)object->data ;
  
  if (flags & TCL_TRACE_READS) {
    /* Read a network short : */
    v = ( (*c)<< 8 ) + ( *(c+1) ) ;
    sprintf(res,"%u",v);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a network short : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in netshort write";
    if (Tcl_GetInt(interp,r,(int *)&v)==TCL_ERROR) return "not a valid int";
    if (v > 0xffff) return "overflow (number too large for a netshort)";
    *c++ = (unsigned char) ( v >> 8 );
    *c   = (unsigned char) ( v & 0xff );
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}

	/*---------*/


/* I/O Short Trace */
char *
Bin_TraceShort(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  short v;

  if (flags & TCL_TRACE_READS) {
    /* Read a short int : */
    memcpy(&v,object->data,sizeof(v));  /* avoid bus error for misalignment */
    sprintf(res,"%d",(int)v);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a short int : */
    char *r;
    int  i;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in short write";
    if (Tcl_GetInt(interp,r,&i)==TCL_ERROR) return "not a valid short";
    v=(short)i;
    memcpy(object->data,&v,sizeof(v));	/* avoid bus error for misalignment */
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}


	/*---------*/


/* I/O Double Trace */
char *
Bin_TraceDouble(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  double v;
  
  if (flags & TCL_TRACE_READS) {
    /* Read a double : */
    memcpy(&v,object->data,sizeof(v));  /* avoid bus error for misalignment */
    Tcl_PrintDouble(interp,v,res);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a double : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in double write";
    if (Tcl_GetDouble(interp,r,&v)==TCL_ERROR) return "not a valid double";
    memcpy(object->data,&v,sizeof(v));	/* avoid bus error for misalignment */
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}

	/*---------*/


#ifdef ADONIS
/* I/O Complex numbers Trace */
char *
Bin_TraceComplex(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  complex v;
  
  if (flags & TCL_TRACE_READS) {
    int lg;
    /* Read a complex : */
    memcpy(&v,object->data,sizeof(v));  /* avoid bus error for misalignment */
    Tcl_PrintDouble(interp,v.real,res);
    lg=strlen(res);
    res[lg]=' ';
    Tcl_PrintDouble(interp,v.imag,res+lg+1);
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a complex : */
    char *r,**argv;
    int argc;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in complex write";
    if (Tcl_SplitList(interp,r,&argc,&argv)==TCL_ERROR) return "not a list";
    if (argc!=2) {
      ckfree((void*)argv);
      return "not a complex {real imag}!";
    }
    if (Tcl_GetDouble(interp,argv[0],&v.real)==TCL_ERROR) {
      ckfree((void*)argv);
      return "real part is not a valid double";
    }
    if (Tcl_GetDouble(interp,argv[1],&v.imag)==TCL_ERROR) {
      ckfree((void*)argv);
      return "imag part is not a valid double";
    }
    memcpy(object->data,&v,sizeof(v));	/* avoid bus error for misalignment */
    ckfree((void*)argv);
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}
#endif

	/*---------*/


/* I/O Float Trace */
char *
Bin_TraceFloat(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  static char res[80];
  float v;
  double d;
  
  if (flags & TCL_TRACE_READS) {
    /* Read a float : */
    memcpy(&v,object->data,sizeof(v));  /* avoid bus error for misalignment */
    d=v;
    Tcl_PrintDouble(interp,d,res);
#ifdef DEBUG
    fprintf(stderr,"f=%f,d=%f,r=%s,s=%d\n",v,d,res,sizeof(v));
#endif
    Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
  } else if (flags & TCL_TRACE_WRITES) {
    /* Write a float : */
    char *r;
    r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
    if (r==NULL) return "null ptr in float write";
    if (Tcl_GetDouble(interp,r,&d)==TCL_ERROR) return "not a valid float";
    v=d;
#ifdef DEBUG
    fprintf(stderr,"f=%f,d=%f,r=%s,s=%d\n",v,d,r,sizeof(v));
#endif
    memcpy(object->data,&v,sizeof(v));	/* avoid bus error for misalignment */
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}

	/*---------*/


/* I/O Ptr Trace */
char *
Bin_TracePtr(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata;
  int sz=object->size;	/* available size (max) */
  
  if (flags & TCL_TRACE_READS) {
    /* special case, char* ptr : */
    if (object->PeekPokeProc==Bin_TraceChar) {
      /* read/get string value : */
      char *buf;
      buf=ckalloc(sz+1); /* for the trailing \0 */
      if (!buf) return "can't allocate string buffer!";
      /* we copy at max all the end of the object */
      strncpy(buf,object->data,sz);
      /* but we add eventually missing 0 */
      buf[sz]='\0';
      Tcl_SetVar2(interp,name1,name2,buf,flags&TCL_GLOBAL_ONLY);
      ckfree(buf);
    } else {
      static char res[80];
      /* Read a ptr : */
      sprintf(res,"%p",object->data);
      Tcl_SetVar2(interp,name1,name2,res,flags&TCL_GLOBAL_ONLY);
    }
  } else if (flags & TCL_TRACE_WRITES) {
    /* special case, char* ptr : */
    if (object->PeekPokeProc==Bin_TraceChar) {
      /* Write a string (must be small enough to fit in object) : */
      char *r;
      r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
      if (r==NULL) return "null ptr in string write";
      strncpy(object->data,r,MIN(sz,strlen(r)+1)); /* silently truncated */
    } else {
      /* Write a ptr : illegal, make it read-only : */
      return "can't change pointers";
    }
  } else {
    /* Unset : */
#ifdef DEBUG
    fprintf(stderr,"\tunset!\n");
#endif
    Bin_DeleteObject(object);
  }
  return NULL;    
}


	/*---------*/


/* I/O String Trace */
char *
Bin_TraceStr(cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* trace flags */
{
  Bin_Object *object=(Bin_Object *)cdata,*of;
  int offset,sz;
  if (flags&TCL_TRACE_UNSETS) 
    {
      /* Unset : */
#ifdef DEBUG
      fprintf(stderr,"\tunset!\n");
#endif
      Bin_DeleteObject(object);
    } 
  else 
    {
      if ((of=object->father)==NULL) return "null father in string op!";
      offset=(((char *)object->data)-((char*)of->data));
      sz=of->size-offset;	/* available size (max) */
#ifdef DEBUG
      fprintf(stderr,"\tof.sz=%d,offs=%d,maxn=%d\n",of->size,offset,sz);
#endif
      if (flags & TCL_TRACE_READS) 
	{
	  char *buf;
	  /* Read a string (must be null terminated! (should verify this)) : */
	  buf=ckalloc(sz+1); /* for the trailing \0 */
	  if (!buf) return "can't allocate string buffer!";
	  /* we copy at max all the end of the object */
	  strncpy(buf,object->data,sz);
	  /* but we add eventually missing 0 */
	  buf[sz]='\0';
	  Tcl_SetVar2(interp,name1,name2,buf,flags&TCL_GLOBAL_ONLY);
	  ckfree(buf);
	}
      else if (flags & TCL_TRACE_WRITES) 
	{
	  /* Write a string (must be small enough to fit in object) : */
	  char *r;
	  r=Tcl_GetVar2(interp,name1,name2,flags&TCL_GLOBAL_ONLY);
	  if (r==NULL) return "null ptr in string write";
	  strncpy(object->data,r,MIN(sz,strlen(r)+1));
	}
    }
  return NULL;    
}

	/*---------*/

/*
 * FreeChild on father unset 
 * internal use
 */
char *
Bin_FreeChild (cdata, interp,name1,name2,flags)
    ClientData cdata;   /* Object pointer */
    Tcl_Interp *interp; /* Current Interpreter */
    char *name1,*name2; /* name of the object */
    int flags;          /* unset flags */
{
  char *name=(char *)cdata;

#ifdef UNSET_DEBUG
  fprintf(stderr,"called unset trace for %s(%s) => unset %s (%p) :",
	 name1,name2 ? name2 : "",name ? name : "<null>",name);
#endif
  if (cdata==NULL) return NULL; /* null data while unsetting child! */
  if (! (flags&TCL_INTERP_DESTROYED)) {
#ifdef UNSET_DEBUG
      if (
#endif
	  Tcl_UnsetVar(interp,name,0) 
#ifdef UNSET_DEBUG
	  ==TCL_ERROR
	  ) {
	  fprintf(stderr,"error!\n");
      } else {
	  fprintf(stderr,"ok!\n");
      }
#else
      ;
#endif
  }
  ckfree(name);
  return NULL;
}

	/*---------*/


#ifndef HAVE_LTOA

char *ltoa(x)
     long x;
{
  static char tmp[21];/* really array  must have size  log10(MAXLONG)+2 */
                      /* this is ok till 64 bits long long */
  *tmp = '\0';
  sprintf(tmp,"%ld",x);
  return(tmp);
}

#endif


	/*---------*/


/*
# For extraction by AutoDoc's c2tcldoc / document

########
doc_section BUGS {
# none known.
}


########
doc_section AUTHOR {
# Laurent Demailly, 
#   <ld@mail.box.eu.org> (pobox <dl@mail.dotcom.fr>) 
# / <URL:http://www.box.eu.org/~dl/>
}
*/

	/*---------*/

#endif  /* End of not for C2MAN section */

